import static org.junit.Assert.assertEquals;

import org.faktorips.runtime.ClassloaderRuntimeRepository;
import org.faktorips.runtime.IProductComponent;
import org.faktorips.runtime.IRuntimeRepository;
import org.faktorips.tutorial.model.hausrat.HausratGrunddeckung;
import org.faktorips.tutorial.model.hausrat.HausratGrunddeckungstyp;
import org.faktorips.tutorial.model.hausrat.HausratProdukt;
import org.faktorips.tutorial.model.hausrat.HausratVertrag;
import org.faktorips.tutorial.model.hausrat.HausratZusatzdeckung;
import org.faktorips.tutorial.model.hausrat.HausratZusatzdeckungstyp;
import org.faktorips.values.Money;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

// tag::test[]

public class TutorialTest {

    private IRuntimeRepository repository;
    private HausratProdukt kompaktProdukt;

    @BeforeEach
    public void setUp() throws Exception {
        // Repository erzeugen
        repository = ClassloaderRuntimeRepository
                .create("org/faktorips/tutorial/produktdaten/internal/faktorips-repository-toc.xml");

        // Referenz auf das Kompaktprodukt aus dem Repository holen
        IProductComponent pc = repository.getProductComponent("hausrat.HR-Kompakt 2019-07");

        // Auf die eigenen Modellklassen casten
        kompaktProdukt = (HausratProdukt) pc;
    }

    @Test
    public void test() {
        System.out.println("Produktname: " + kompaktProdukt.getProduktname());
        System.out.println("Vorschlag Vs pro 1qm: " + kompaktProdukt.getVorschlagVersSummeProQm());
        System.out.println("Default Zahlweise: " + kompaktProdukt.getDefaultValueZahlweise());
        System.out.println("Erlaubte Zahlweisen: " + kompaktProdukt.getAllowedValuesForZahlweise());
        System.out.println("Default Vs: " + kompaktProdukt.getDefaultValueVersSumme());
        System.out.println("Bereich Vs: " + kompaktProdukt.getAllowedValuesForVersSumme());
        System.out.println("Default Wohnflaeche: " + kompaktProdukt.getDefaultValueWohnflaeche());
        System.out.println("Bereich Wohnflaeche: " + kompaktProdukt.getAllowedValuesForWohnflaeche());
    }
    // end::test[]

    // tag::testGetTarifzone[]

    @Test
    public void testGetTarifzone() {
        // Erzeugen eines Hausratvertrags mit der Factorymethode des Produktes
        HausratVertrag vertrag = kompaktProdukt.createHausratVertrag();
        vertrag.setPlz("45525");
        assertEquals("III", vertrag.getTarifzone());
    }
    // end::testGetTarifzone[]

    // tag::testBerechneBeitrag[]
    @Test
    public void testBerechneBeitrag() {
        // Erzeugen eines hausratvertrags mit der Factorymethode des Produktes
        HausratVertrag vertrag = kompaktProdukt.createHausratVertrag();
        // Vertragsattribute setzen
        vertrag.setPlz("45525"); // => tarifzone 3
        vertrag.setVersSumme(Money.euro(60_000));
        vertrag.setZahlweise(2); // halbjaehrlich
        /*
         * Grunddeckungstyp holen, der dem Produkt in der Anpassungsstufe zugeordnet
         * ist.
         */
        HausratGrunddeckungstyp deckungstyp = kompaktProdukt.getHausratGrunddeckungstyp();
        // Grunddeckung erzeugen und zum Vertag hinzufuegen
        HausratGrunddeckung deckung = vertrag.newHausratGrunddeckung(deckungstyp);
        // Beitrag berechnen und Ergebniss pruefen
        vertrag.berechneBeitrag();

        // tarifzone 3 => beitragssatz = 1.21 jahresbasisbeitrag
        // = versicherungsumme / 1000 * beitragssatz = 60000 / 1000 * 1,21
        // = 72,60
        assertEquals(Money.euro(72, 60), deckung.getJahresbasisbeitrag());

        // Jahresbasisbeitrag vertrag = Jahresbasisbeitrag deckung
        assertEquals(Money.euro(72, 60), vertrag.getJahresbasisbeitrag());

        // NettobeitragZw = 72,60 / 2 * 1,03 (wg. Ratenzahlungszuschlag von 3%)
        // = 37,389
        // => 37,39
        assertEquals(Money.euro(37, 39), vertrag.getNettobeitragZw());

        // BruttobeitragZw = 37,39 * Versicherungssteuersatz = 37,39 * 1,19
        // = 44,49
        assertEquals(Money.euro(44, 49), vertrag.getBruttobeitragZw());
    }
    // end::testBerechneBeitrag[]

    // tag::testBerechneJahresbasisbeitragFahrraddiebstahl[]
    @Test
    public void testBerechneJahresbasisbeitragFahrraddiebstahl() {
        // Erzeugen eines hausratvertrags mit der Factorymethode des Produktes
        HausratVertrag vertrag = kompaktProdukt.createHausratVertrag();

        // Vertragsattribute setzen
        vertrag.setVersSumme(Money.euro(60_000));

        /*
         * Zusatzdeckungstyp Fahrraddiebstahl holen. Der Einfachheit halber nehmen wir
         * hier an, der erste Zusatzdeckungstyp ist Fahrraddiebstahl
         */
        HausratZusatzdeckungstyp deckungstyp = kompaktProdukt.getHausratZusatzdeckungstyp(0);

        // Zusatzdeckung erzeugen
        HausratZusatzdeckung deckung = vertrag.newHausratZusatzdeckung(deckungstyp);

        // Jahresbasisbeitrag berechnen
        Money jahresbasisbeitrag = deckung.berechneJahresbasisbeitrag();

        /*
         * Jahresbasisbeitrag testen
         * 
         * Versicherungssumme der Deckung = 1% von 60.000, max 5.000 => 600 Beitrag =
         * 10% von 600 = 60
         */
        assertEquals(Money.euro(60, 0), jahresbasisbeitrag);
    }
    // end::testBerechneJahresbasisbeitragFahrraddiebstahl[]

}
