import static org.junit.jupiter.api.Assertions.assertEquals;

import org.faktorips.runtime.ClassloaderRuntimeRepository;
import org.faktorips.runtime.IProductComponent;
import org.faktorips.runtime.IRuntimeRepository;
import org.faktorips.tutorial.model.home.HomeBaseCoverage;
import org.faktorips.tutorial.model.home.HomeBaseCoverageType;
import org.faktorips.tutorial.model.home.HomeContract;
import org.faktorips.tutorial.model.home.HomeExtraCoverage;
import org.faktorips.tutorial.model.home.HomeExtraCoverageType;
import org.faktorips.tutorial.model.home.HomeProduct;
import org.faktorips.values.Money;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

//tag::test[]

public class TutorialTest {

    private IRuntimeRepository repository;
    private HomeProduct compactProduct;

    @BeforeEach
    public void setUp() throws Exception {
        // Repository erzeugen
        repository = ClassloaderRuntimeRepository
                .create("org/faktorips/tutorial/productdata/internal/faktorips-repository-toc.xml");

        // Referenz auf das compactProduct aus dem Repository holen
        IProductComponent pc = repository.getProductComponent("home.HC-Compact 2021-12");

        // Auf die eigenen Modellklassen casten
        compactProduct = (HomeProduct) pc;
    }

    @Test
    public void test() {
        System.out.println("Product name: " + compactProduct.getProductname());
        System.out.println("Proposed sum insured per sqm: " + compactProduct.getProposedSumInsuredPerSqm());
        System.out.println("Default modes of payment: " + compactProduct.getDefaultValuePaymentMode());
        System.out.println("Allowed modes of payment: " + compactProduct.getAllowedValuesForPaymentMode());
        System.out.println("Default sum insured: " + compactProduct.getDefaultValueSumInsured());
        System.out.println("Range sum insured: " + compactProduct.getAllowedValuesForSumInsured());
        System.out.println("Default living space: " + compactProduct.getDefaultValueLivingSpace());
        System.out.println("Range living space: " + compactProduct.getAllowedValuesForLivingSpace());
    }
    // end::test[]

    // tag::testGetRatingDistrict[]
    @Test
    public void testGetRatingDistrict() {
        // Create a new HomeContract with the products factory method
        HomeContract contract = compactProduct.createHomeContract();
        contract.setZipcode("45525");
        assertEquals("III", contract.getRatingDistrict());
    }
    // end::testGetTarifzone[]

    // tag::testComputePremium[]
    @Test
    public void testComputePremium() {
        // Create a new HomeContract with the products factory method
        HomeContract contract = compactProduct.createHomeContract();
        // Set contract attributes
        contract.setZipcode("45525");
        contract.setSumInsured(Money.euro(60_000));
        contract.setPaymentMode(2);
        // Get the base coverage type that is assigned to the product
        HomeBaseCoverageType coverageType = compactProduct.getHomeBaseCoverageType();
        // Create the base coverage and add it to the contract
        HomeBaseCoverage coverage = contract.newHomeBaseCoverage(coverageType);
        // Compute the premium and check the results
        contract.computePremium();
        // rating district III => premiumRate = 1,21 annualBasePremium = sumInsured /
        // 1000 * premiumRate = 60000 / 1000 * 1,21 = 72,60
        assertEquals(Money.euro(72, 60), coverage.getAnnualBasePremium());
        // contract.annualBasePremium = baseCoverage.annualBasePremium
        assertEquals(Money.euro(72, 60), contract.getAnnualBasePremium());
        // netPremiumPm = 72,60 / 2 * 1,03 (semi-annual, 3% surcharge) = 37,389 => 37,39
        assertEquals(Money.euro(37, 39), contract.getNetPremiumPm());
        // grossPremiumPm = 37,39 * taxMultiplier = 37,39 * 1,19 = 44,49
        assertEquals(Money.euro(44, 49), contract.getGrossPremiumPm());
    }
    // end::testComputePremium[]

    // tag::testComputePremiumBicycleTheft[]
    @Test
    public void testComputePremiumBicycleTheft() {
        // Create a new HomeContract with the products factory method
        HomeContract contract = compactProduct.createHomeContract();
        // Set contract attributes
        contract.setSumInsured(Money.euro(60_000));
        // Get ExtraCoveragetype BicycleTheft
        // (To make it easy, we assume that this is the first one)
        HomeExtraCoverageType coverageType = compactProduct.getHomeExtraCoverageType(0);
        // Create extra coverage
        HomeExtraCoverage coverage = contract.newHomeExtraCoverage(coverageType);
        // compute AnnualBasePremium and check it
        coverage.computeAnnualBasePremium();
        // Coverage.SumInsured = 1% from 60,0000, max 5.000 => 600
        // Premium = 10% from 600 = 60
        assertEquals(Money.euro(60, 0), coverage.computeAnnualBasePremium());
    }
    // end::testComputePremiumBicycleTheft[]
}