package org.faktorips.tutorial.de.productdataprovider;

import java.text.SimpleDateFormat;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Properties;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.faktorips.runtime.IRuntimeRepository;
import org.faktorips.runtime.IRuntimeRepositoryManager;
import org.faktorips.runtime.formula.groovy.GroovyFormulaEvaluatorFactory;
import org.faktorips.runtime.productdataprovider.DataModifiedRuntimeException;
import org.faktorips.runtime.productdataprovider.DetachedContentRuntimeRepositoryManager;
import org.faktorips.runtime.productdataprovider.ejbclient.EjbProductDataProviderFactory;
import org.faktorips.tutorial.hausrat.hausrat.IHausratProdukt;
import org.faktorips.tutorial.hausrat.hausrat.IHausratProduktGen;
import org.faktorips.tutorial.hausrat.hausrat.IHausratVertrag;
import org.faktorips.tutorial.hausrat.hausrat.IHausratZusatzdeckungstyp;
import org.faktorips.tutorial.modell.base.IDeckung;
import org.faktorips.values.Money;

public class ProductDataProviderDemo {

	/**
	 * Der Repository Manager wird in einer final Variable gehalten.
	 */
	public final IRuntimeRepositoryManager repositoryManager;

	/**
	 * Der Konstruktor initialisiert den Repository Manager.
	 * 
	 * @throws NamingException
	 *             Kann beim initialisieren des {@link InitialContext} auftreten
	 */
	public ProductDataProviderDemo() throws NamingException {
		// Die Properties sind nur für OpenEjb, für andere Application Server
		// müssen evtl. andere Properties gesetzt werden
		Properties properties = new Properties();
		properties.put(Context.INITIAL_CONTEXT_FACTORY,
				"org.apache.openejb.client.RemoteInitialContextFactory");
		properties.put(Context.PROVIDER_URL,"ejbd://127.0.0.1:4201");

		InitialContext initialContext = new InitialContext(properties);
		EjbProductDataProviderFactory pdpFactory = new EjbProductDataProviderFactory(
				"HausratProductDataProviderDemo", initialContext);

		repositoryManager = new DetachedContentRuntimeRepositoryManager.Builder(
				pdpFactory).setFormulaEvaluatorFactory(
				new GroovyFormulaEvaluatorFactory()).build();
	}

	public static void main(String[] args) {
		try {
			ProductDataProviderDemo pdpDemo = new ProductDataProviderDemo();

			try {
				// Neuen HausratVertrag erstellen
				IHausratVertrag vertrag = pdpDemo.createHausratVertrag();
				
				// Berechnen der Werte
				pdpDemo.calculateVertrag(vertrag);
				
				// Ausgeben des Vertrags
				pdpDemo.printVertrag(vertrag);
			} catch (DataModifiedRuntimeException e) {
				System.err.println("Daten wurden geändert! Abfrage wurde abgebrochen.");
			}
		} catch (NamingException e) {
			e.printStackTrace();
		}
	}

	private IHausratVertrag createHausratVertrag() {
		IRuntimeRepository repository = repositoryManager
				.getActualRuntimeRepository();
		IHausratProdukt produkt = (IHausratProdukt) repository
				.getProductComponent("hausrat.HR-Kompakt 2008-04");
		IHausratVertrag vertrag = produkt.createHausratVertrag();
		vertrag.setPlz("12345");
		vertrag.setVersSumme(Money.euro(50000));
		vertrag.setWirksamAb(new GregorianCalendar());
		vertrag.setWohnflaeche(80);
		vertrag.setZahlweise(1);

		IHausratProduktGen produktGen = (IHausratProduktGen) produkt
				.getLatestProductComponentGeneration();
		vertrag.newHausratGrunddeckung(produktGen.getHausratGrunddeckungstyp());

		List<IHausratZusatzdeckungstyp> zusatzdeckungsTypen = produktGen
				.getHausratZusatzdeckungstypen();
		for (IHausratZusatzdeckungstyp zusatzdeckungstyp : zusatzdeckungsTypen) {
			vertrag.newHausratZusatzdeckung(zusatzdeckungstyp);
		}
		return vertrag;
	}
	
	private void calculateVertrag(IHausratVertrag vertrag) {
		vertrag.ermittleVorschlagVersicherungssumme();
		for (IDeckung deckung : vertrag.getDeckungen()) {
			// hier wird intern an den Groovy Interpreter delegiert
			deckung.berechneJahresbasisbeitrag();
		}
	}

	private void printVertrag(IHausratVertrag vertrag) {
		System.out.println("---------------------------------");
		System.out.println("Angebot");
		System.out.println("Produkt:\t" + vertrag.getProductComponent());
		System.out.println("PLZ:\t\t" + vertrag.getPlz());
		System.out.println("VersSumme:\t" + vertrag.getVersSumme());
		System.out.println("WirksamAb:\t" + SimpleDateFormat.getInstance().format(vertrag.getWirksamAb().getTime()));
		System.out.println("Wohnfläche:\t" + vertrag.getWohnflaeche());
		System.out.println("Zahlweise:\t" + vertrag.getZahlweise());
		System.out.println("Vorschlag:\t" + vertrag.getVorschlagVersSumme());
		System.out.println("Deckungen:");
		for (IDeckung deckung : vertrag.getDeckungen()) {
			System.out.println("\t" + deckung.getDeckungstyp().getId());
			System.out.println("\t\tJahresbeitrag:\t" + deckung.getJahresbasisbeitrag());
		}
		System.out.println("---------------------------------");
	}

}
