/*******************************************************************************
 * Copyright (c) 2005-2010 Faktor Zehn AG und andere.
 * 
 * Alle Rechte vorbehalten.
 * 
 * Dieses Programm und alle mitgelieferten Sachen (Dokumentationen, Beispiele, Konfigurationen,
 * etc.) duerfen nur unter den Bedingungen der Faktor-Zehn-Community Lizenzvereinbarung - Version
 * 0.1 (vor Gruendung Community) genutzt werden, die Bestandteil der Auslieferung ist und auch unter
 * http://www.faktorzehn.org/fips:lizenz eingesehen werden kann.
 * 
 * Mitwirkende: Faktor Zehn AG - initial API and implementation - http://www.faktorzehn.de
 *******************************************************************************/

package org.faktorips.runtime.productdataprovider.ejbclient;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.StringReader;

import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.faktorips.productdataservice.IProductDataService;
import org.faktorips.productdataservice.XmlTimestampData;
import org.faktorips.runtime.IVersionChecker;
import org.faktorips.runtime.internal.DateTime;
import org.faktorips.runtime.internal.toc.EnumContentTocEntry;
import org.faktorips.runtime.internal.toc.GenerationTocEntry;
import org.faktorips.runtime.internal.toc.IReadonlyTableOfContents;
import org.faktorips.runtime.internal.toc.ProductCmptTocEntry;
import org.faktorips.runtime.internal.toc.ReadonlyTableOfContents;
import org.faktorips.runtime.internal.toc.TableContentTocEntry;
import org.faktorips.runtime.internal.toc.TestCaseTocEntry;
import org.faktorips.runtime.productdataprovider.AbstractProductDataProvider;
import org.faktorips.runtime.productdataprovider.DataModifiedException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

/**
 * This product data provider loads the product data content from an ejb 3.0 stateless session bean.
 * The service is called for every request of product data. The service provides the xml content
 * with the version of the actually loaded toc. This product data provider checks whether the table
 * of content loaded by the client is the same as used by the service. When the version differs, a
 * {@link DataModifiedException} is thrown by any method providing product data. To load the new
 * product data you have to reload the table of content by calling {@link #getToc()}.
 * 
 * @author dirmeier
 */
public class EjbProductDataProvider extends AbstractProductDataProvider {

    public final IProductDataService productDataService;

    /**
     * In case that multiple threads have access to this provider, the version should be the same
     * for all threads. So this variable have to be volatile.
     */
    public final String version;

    private final ReadonlyTableOfContents toc;

    /**
     * Initializing the product data provider getting information from the stateless session bean
     * <code>org.faktorips.productdataservice.ProductDataService</code>
     * 
     * The properties are used to initiate the {@link InitialContext} you have to set at least the
     * factory by property key Context.INITIAL_CONTEXT_FACTORY
     * 
     */
    EjbProductDataProvider(String beanName, InitialContext initialContext, IVersionChecker versionChecker) {
        super(versionChecker);
        try {
            String sessionBenaName = beanName;
            productDataService = (IProductDataService)initialContext.lookup(sessionBenaName);
            version = getBaseVersion();
            toc = loadToc();
        } catch (NamingException e) {
            throw new RuntimeException(e);
        }
    }

    private ReadonlyTableOfContents loadToc() {
        XmlTimestampData timestampData = productDataService.getTocData();
        try {
            Document doc = getDocumentBuilder().parse(new InputSource(new StringReader(timestampData.xmlData)));
            Element tocElement = doc.getDocumentElement();
            ReadonlyTableOfContents toc = new ReadonlyTableOfContents();
            toc.initFromXml(tocElement);
            return toc;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public String getVersion() {
        return version;
    }

    @Override
    public InputStream getEnumContentAsStream(EnumContentTocEntry tocEntry) throws DataModifiedException {
        XmlTimestampData timestampData = productDataService.getEnumContent(tocEntry.getImplementationClassName());
        throwExceptionIfExpired(tocEntry.getImplementationClassName(), timestampData.version);
        return new ByteArrayInputStream(timestampData.xmlData.getBytes());
    }

    @Override
    public Element getProductCmptData(ProductCmptTocEntry tocEntry) throws DataModifiedException {
        XmlTimestampData timestampData = productDataService.getProductCmptData(tocEntry.getIpsObjectId());
        throwExceptionIfExpired(tocEntry.getIpsObjectId(), timestampData.version);
        try {
            Document doc = getDocumentBuilder().parse(new InputSource(new StringReader(timestampData.xmlData)));
            return doc.getDocumentElement();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Element getProductCmptGenerationData(GenerationTocEntry tocEntry) throws DataModifiedException {
        Element docElement = getProductCmptData(tocEntry.getParent());
        NodeList nl = docElement.getChildNodes();
        DateTime validFrom = tocEntry.getValidFrom();
        for (int i = 0; i < nl.getLength(); i++) {
            if ("Generation".equals(nl.item(i).getNodeName())) {
                Element genElement = (Element)nl.item(i);
                DateTime generationValidFrom = DateTime.parseIso(genElement.getAttribute("validFrom"));
                if (validFrom.equals(generationValidFrom)) {
                    return genElement;
                }
            }
        }
        throw new RuntimeException("Can't find the generation for the toc entry " + tocEntry);
    }

    @Override
    public InputStream getTableContentAsStream(TableContentTocEntry tocEntry) throws DataModifiedException {
        XmlTimestampData timestampData = productDataService.getTableContent(tocEntry.getIpsObjectQualifiedName());
        throwExceptionIfExpired(tocEntry.getIpsObjectQualifiedName(), timestampData.version);
        return new ByteArrayInputStream(timestampData.xmlData.getBytes());
    }

    @Override
    public Element getTestcaseElement(TestCaseTocEntry tocEntry) throws DataModifiedException {
        XmlTimestampData timestampData = productDataService.getTestCaseData(tocEntry.getIpsObjectQualifiedName());
        throwExceptionIfExpired(tocEntry.getIpsObjectId(), timestampData.version);
        try {
            Document doc = getDocumentBuilder().parse(new InputSource(new StringReader(timestampData.xmlData)));
            return doc.getDocumentElement();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public synchronized IReadonlyTableOfContents getToc() {
        return toc;
    }

    @Override
    public String getBaseVersion() {
        return productDataService.getProductDataVersion();
    }

    private void throwExceptionIfExpired(String ipsObject, String version) throws DataModifiedException {
        if (!getVersionChecker().isCompatibleVersion(getVersion(), version)) {
            DataModifiedException e = new DataModifiedException(MODIFIED_EXCEPTION_MESSAGE + ipsObject, this.version,
                    version);
            throw e;
        }
    }

}
