/*******************************************************************************
 * Copyright (c) 2005-2010 Faktor Zehn AG und andere.
 * 
 * Alle Rechte vorbehalten.
 * 
 * Dieses Programm und alle mitgelieferten Sachen (Dokumentationen, Beispiele, Konfigurationen,
 * etc.) duerfen nur unter den Bedingungen der Faktor-Zehn-Community Lizenzvereinbarung - Version
 * 0.1 (vor Gruendung Community) genutzt werden, die Bestandteil der Auslieferung ist und auch unter
 * http://www.faktorzehn.org/fips:lizenz eingesehen werden kann.
 * 
 * Mitwirkende: Faktor Zehn AG - initial API and implementation - http://www.faktorzehn.de
 *******************************************************************************/

package org.faktorips.runtime.formula.groovy;

import groovy.lang.Binding;
import groovy.lang.GroovyShell;
import groovy.lang.Script;

import java.util.List;

import org.faktorips.runtime.IProductComponentGeneration;
import org.faktorips.runtime.formula.AbstractFormulaEvaluator;

/**
 * This formula evaluator evaluates the formulas using groovy. The implementation expects the
 * formula compiled as java code. Each compiled expression should be a complete java method
 * including the method signature. It expects the product component generation in which context the
 * method should run, is referenced as 'this' within the method. That means the method is expected
 * to run within the context of the product component generation.
 * 
 * @author dirmeier
 */
public class GroovyFormulaEvaluator extends AbstractFormulaEvaluator {

    public static final String THIS_CLASS_VAR = "thizProductCmptGeneration"; //$NON-NLS-1$

    private final Binding binding = new Binding();

    private final Script groovyScript;

    /**
     * The constructor expects the product component generation and a list of java method, one for
     * every formula that should be evaluated by this formula evaluator. The methods should be
     * designed to run in the context of the product component generation. The product component
     * generation should be referenced as 'this' whithin the methods body.
     * 
     * @param gen the product component generation in which context the compiled epxression methods
     *            would run
     * @param compiledExpressions a list of compiled expression methods, one for every formula
     */
    public GroovyFormulaEvaluator(IProductComponentGeneration gen, List<String> compiledExpressions) {
        super(gen);
        binding.setVariable(THIS_CLASS_VAR, gen);
        GroovyShell groovyShell = new GroovyShell(binding);
        String sourceCode = getSourceCode(compiledExpressions);
        groovyScript = groovyShell.parse(sourceCode);
    }

    private String getSourceCode(List<String> formulaList) {
        StringBuilder sourceCode = new StringBuilder();
        for (String formula : formulaList) {
            formula = formula.replaceAll("this", THIS_CLASS_VAR);
            sourceCode.append(formula).append('\n');
        }
        return sourceCode.toString();
    }

    @Override
    protected Object evaluateInternal(String formularName, Object... parameters) {
        return groovyScript.invokeMethod(formularName, parameters);
    }

}
