/*******************************************************************************
 * Copyright (c) 2005-2010 Faktor Zehn AG und andere.
 * 
 * Alle Rechte vorbehalten.
 * 
 * Dieses Programm und alle mitgelieferten Sachen (Dokumentationen, Beispiele, Konfigurationen,
 * etc.) duerfen nur unter den Bedingungen der Faktor-Zehn-Community Lizenzvereinbarung - Version
 * 0.1 (vor Gruendung Community) genutzt werden, die Bestandteil der Auslieferung ist und auch unter
 * http://www.faktorzehn.org/fips:lizenz eingesehen werden kann.
 * 
 * Mitwirkende: Faktor Zehn AG - initial API and implementation - http://www.faktorzehn.de
 *******************************************************************************/

package org.faktorips.productdataservice;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.faktorips.runtime.internal.toc.ReadonlyTableOfContents;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * The TocProvider loads the table of content and provide it as object of
 * {@link ReadonlyTableOfContents} or as content data of type {@link String}. You also could get the
 * modification time of the loaded toc. Once loaded the table of content will never be reloaded! We
 * assume that the service using these classes creates new instances after deploying new product
 * data.
 * 
 * @author dirmeier
 */
public class TocProvider {

    private transient final ReadonlyTableOfContents toc;

    private transient final String tocData;

    public TocProvider(URL tocUrl) {
        DocumentBuilder docBuilder = createDocumentBuilder();
        toc = loadToc(tocUrl, docBuilder);
        tocData = loadTocData(tocUrl);
    }

    /**
     * Get the table of content
     * 
     * @return the table of content
     */
    public ReadonlyTableOfContents getTableOfContents() {
        return toc;
    }

    /**
     * Get the string content of the table of content
     * 
     * @return the XML representation of the table of content
     */
    public String getTableOfContentsData() {
        return tocData;
    }

    /**
     * Getting the last modification time stamp of the already loaded table of content or 0 if no
     * table of content is loaded. Does not look for modifications in file because we assume that
     * this instance is recreated after deploying new product data
     */
    public String getProductDataVersion() {
        return toc.getProductDataVersion();
    }

    private final DocumentBuilder createDocumentBuilder() {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setNamespaceAware(false);
        DocumentBuilder builder;
        try {
            builder = factory.newDocumentBuilder();
        } catch (ParserConfigurationException e1) {
            throw new RuntimeException("Error creaing document builder.", e1);
        }
        return builder;
    }

    private ReadonlyTableOfContents loadToc(URL tocUrl, DocumentBuilder docBuilder) {
        InputStream is = null;
        try {
            is = tocUrl.openStream();
            Document doc = docBuilder.parse(is);
            Element tocElement = doc.getDocumentElement();
            ReadonlyTableOfContents newToc = new ReadonlyTableOfContents();
            newToc.initFromXml(tocElement);
            return newToc;
        } catch (Exception e) {
            throw new RuntimeException("Error reading contents from " + tocUrl, e);
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    throw new RuntimeException("Error closing stream of " + tocUrl.getFile(), e);
                }
            }
        }
    }

    private String loadTocData(URL tocUrl) {
        try {
            String newTocData = FileReaderUtil.readContent(tocUrl);
            return newTocData;
        } catch (Exception e) {
            throw new RuntimeException("Error reading contents from " + tocUrl.getFile(), e);
        }
    }

}
