/*******************************************************************************
 * Copyright (c) 2005-2010 Faktor Zehn AG und andere.
 * 
 * Alle Rechte vorbehalten.
 * 
 * Dieses Programm und alle mitgelieferten Sachen (Dokumentationen, Beispiele, Konfigurationen,
 * etc.) duerfen nur unter den Bedingungen der Faktor-Zehn-Community Lizenzvereinbarung - Version
 * 0.1 (vor Gruendung Community) genutzt werden, die Bestandteil der Auslieferung ist und auch unter
 * http://www.faktorzehn.org/fips:lizenz eingesehen werden kann.
 * 
 * Mitwirkende: Faktor Zehn AG - initial API and implementation - http://www.faktorzehn.de
 *******************************************************************************/

package org.faktorips.productdataservice;

import java.io.IOException;
import java.net.URL;

import javax.annotation.Resource;
import javax.ejb.Stateless;

import org.faktorips.runtime.internal.toc.ProductCmptTocEntry;
import org.faktorips.runtime.internal.toc.TocEntryObject;

/**
 * An EJB 3.0 stateless session bean providing product data. The table of content is loaded by the
 * {@link TocProvider} using the context {@link ClassLoader} of {@link Thread#currentThread()} to
 * find the product data resources. Once the table of content is loaded we never reload and do not
 * look for file changes. We assume that the application server creates new instances after
 * deploying new product data.
 * 
 * @author dirmeier
 */
@Stateless
public class ProductDataService implements IProductDataService {

    public static final String TOCFILE_PROPERTY_KEY = "org.faktorips.pds.tocfile";

    // Have to be volatile to use the double-check ideom @see Item 71 in Effective Java (Second
    // Editions)
    private volatile transient TocProvider tocProvider;

    private transient final ClassLoader classLoader = Thread.currentThread().getContextClassLoader();

    @Resource
    private String tocFileName;

    /**
     * @return Returns the tocProvider.
     */
    private TocProvider getTocProvider() {
        // This local variable is for performance issues @see Item 71 of Effective Java (Second
        // Editions)
        // we use the double-check ideom here
        TocProvider result = tocProvider;
        if (result == null) {
            synchronized (this) {
                result = tocProvider;
                if (result == null) {
                    if (tocFileName == null) {
                        throw new IllegalArgumentException("You have to set the tocFileName in your EJB environment.");
                    }
                    tocProvider = result = new TocProvider(classLoader.getResource(tocFileName));
                }
            }
        }
        return result;
    }

    @Override
    public XmlTimestampData getTocData() {
        return new XmlTimestampData(getTocProvider().getTableOfContentsData(), getTocProvider().getProductDataVersion());
    }

    @Override
    public String getProductDataVersion() {
        return getTocProvider().getProductDataVersion();
    }

    @Override
    public XmlTimestampData getProductCmptData(String ipsObjectId) {
        ProductCmptTocEntry entry = getTocProvider().getTableOfContents().getProductCmptTocEntry(ipsObjectId);
        return getDataForTocEntry(entry);
    }

    @Override
    public XmlTimestampData getTestCaseData(String qName) {
        TocEntryObject entry = getTocProvider().getTableOfContents().getTestCaseTocEntryByQName(qName);
        return getDataForTocEntry(entry);
    }

    @Override
    public XmlTimestampData getTableContent(String qName) {
        TocEntryObject entry = getTocProvider().getTableOfContents().getTableTocEntryByQualifiedTableName(qName);
        return getDataForTocEntry(entry);
    }

    @Override
    public XmlTimestampData getEnumContent(String className) {
        TocEntryObject entry = getTocProvider().getTableOfContents().getEnumContentTocEntry(className);
        return getDataForTocEntry(entry);
    }

    private XmlTimestampData getDataForTocEntry(TocEntryObject entry) {
        if (entry == null) {
            return null;
        }
        String xmlResourceName = entry.getXmlResourceName();
        URL url = classLoader.getResource(xmlResourceName);
        if (url == null) {
            throw new RuntimeException("Toc file containts invalid xml resource name: " + entry.getXmlResourceName());
        }
        String content;
        try {
            content = FileReaderUtil.readContent(url);
        } catch (IOException e) {
            throw new RuntimeException("Error reading contents from " + url.getFile(), e);
        }
        return new XmlTimestampData(content, getProductDataVersion());
    }
}
